#! /usr/bin/env python

"""
This script loads firmware onto the network tester using the EFM bootloader.
"""

import logging
import argparse
from multiprocessing.pool import ThreadPool
from functools import partial
from time import sleep
from xmodem import XMODEM, CRC
from serial import Serial
from serial.tools import list_ports

LOG = logging.getLogger(__name__)

def upload_firmware(dev_path, firmware_path):
    """
    Opens the serial device at `dev_path` and uploads the firmware
    binary at `firmware_path`.
    """
    LOG.info("Opening %s", dev_path)
    with Serial(dev_path, timeout=0.5) as dev, open(firmware_path, 'rb') as firmware:

        def getc(size, timeout=1):
            dev.timeout = timeout
            return dev.read(size)

        def putc(data, timeout=1):
            dev.write_timeout = timeout
            return dev.write(data)

        dev.write(b'U')
        flushed = dev.readlines()
        LOG.debug(flushed)
        dev.write(b'u')
        flushed = dev.readlines()
        LOG.debug(flushed)
        sleep(0.5)

        LOG.info("Starting XMODEM transfer for %s", dev_path)
        xmodem = XMODEM(getc, putc)
        if xmodem.send(firmware):
            LOG.info("Success in xmodem transfer for %s", dev.port)
        else:
            LOG.error("Error in xmodem transfer for %s", dev.port)

def main():
    parser = argparse.ArgumentParser(description=__doc__)
    parser.add_argument('firmware', help='path to the firmware to load')
    parser.add_argument('--device', help='Path of the device to program. \
        Default is to find CP210x devices and upload to all of them.')
    parser.add_argument('--verbose', '-v', action='store_true')
    args = parser.parse_args()

    logging.basicConfig(level=logging.DEBUG if args.verbose else logging.INFO)

    if args.device:
        upload_firmware(args.device, args.firmware)
    else:
        devs = [dev for [dev, _, _] in list_ports.grep('CP210.')]
        upload_with_firmware = partial(upload_firmware, firmware_path=args.firmware)
        pool = ThreadPool(len(devs))
        pool.map(upload_with_firmware, devs)


if __name__ == '__main__':
    main()
