
#include <stdbool.h>
#include <stdint.h>
#include <stdio.h>

#include "FreeRTOS.h"
#include "task.h"

#include "lte_ifc.h"
#include "lte_module.h"

#include "debug_print.h"

static uint32_t s_pkt_id_of_mailbox_request = 0xFFFFFFFF;

static void print_lte_ifc_error(char *label, int32_t ret_val)
{
    if (ret_val < 0)
    {
        debug_printf("LTE IFC ERROR(%s) : %d : %s\n", label, ret_val, lte_ifc_ack_desc(ret_val));
    }
}

int32_t lte_module_init(void)
{
    lte_ifc_hal_tty_open(NULL, 0);
    
    lte_version_t ver;
    int32_t i32_ret = lte_ifc_version_get(&ver);
    print_lte_ifc_error("lte_ifc_version_get", i32_ret);
    if (i32_ret == LTE_IFC_ACK)
    {
        debug_printf("Link Labs Module Version: %d.%d.%d\n", ver.major, ver.minor, ver.tag);
    }
    return i32_ret;
}

int32_t lte_module_reset(void)
{
    int32_t i32_ret;
    do {
        i32_ret = lte_ifc_reset_mcu();
        vTaskDelay(1000 / portTICK_PERIOD_MS);
    } while (i32_ret != 0);
    return i32_ret;
}

static void print_all_irq_flag_descs(uint32_t irq_flags_vec)
{
    for (uint32_t n=0; n<32; n++)
    {
        uint32_t single_bit = 1 << n;
        if (irq_flags_vec & single_bit)
        {
            debug_printf("%s ", lte_ifc_irq_flag_desc(single_bit));
        }
    }
    // Newline only if we printed something
    if (irq_flags_vec != 0)
    {
        debug_printf("\n");
    }
}

int32_t lte_module_get_irq_flags(uint32_t* p)
{
    uint32_t irq_flags_to_clear = 0xFFFFFFFF;
    uint32_t irq_flags_read = 0;
    
    // Check what flags are set, and clear some if we want
    int32_t i32_ret = lte_ifc_irq_flags(irq_flags_to_clear, &irq_flags_read);
    if (i32_ret < 0)
    {
        print_lte_ifc_error("lte_irq_flags", i32_ret);
    }
    else
    {
        // debug_printf("irq_flags = 0x%08X\n", irq_flags_read);
        print_all_irq_flag_descs(irq_flags_read);
        *p = irq_flags_read;
        
        if (irq_flags_read & IRQ_FLAGS_ASSERT)
        {
            static char filename[30];
            uint32_t line = 0;
            uint32_t uptime_sec = 0;
            
            int32_t i32_ret2 = lte_ifc_get_assert_info(&filename[0], &line, &uptime_sec);
            if (i32_ret2 < 0)
            {
                print_lte_ifc_error("lte_ifc_get_assert_info", i32_ret2);
            }
            else
            {
                debug_printf("Module Assert! %s, Line: %d, Uptime: %d [s]\n", filename, line, uptime_sec);
            }
        }
    }
    
    return i32_ret;
}

int32_t lte_module_send_uplink(const uint8_t* buf, uint16_t len)
{
    uint32_t packet_id = 0;
    
    int32_t i32_ret = lte_ifc_message_send((uint8_t *) buf, len, 0, &packet_id);
    if (i32_ret != -104)
    {
        print_lte_ifc_error("lte_ifc_message_send", i32_ret);
    }    
    if (i32_ret == LTE_IFC_ACK)
    {
        debug_printf("lte_module_send_uplink: packet_id = %d\n", packet_id);
    }
    return i32_ret;
}

int32_t lte_module_get_uplink_status(lte_task_config_t* p_task_cfg)
{
    // Get uplink status from module, there might be more than 1
    uint8_t new_status = 0;
    uint32_t packet_id = 0;
    int32_t packet_status = 0;
    uint32_t packet_time_in_queue_ms = 0;
    
    do 
    {
        new_status = 0;
        packet_id = 0;
        packet_status = 0;
        packet_time_in_queue_ms = 0;

        int32_t i32_ret = lte_ifc_message_status(&new_status, &packet_id, &packet_status, &packet_time_in_queue_ms);
        if (i32_ret < 0)
        {
            print_lte_ifc_error("lte_ifc_message_status", i32_ret);
        }
        else
        {
            if (new_status == 1)
            {
                #if 1 // Debug only
                const char* s = lte_ifc_ack_desc(packet_status);
                debug_printf("lte_ifc_message_status: packet_id = %d, status = %d (%s), time_in_queue = %d [ms]\n",
                    packet_id, packet_status, s, packet_time_in_queue_ms);
  
                static uint32_t s_tx_done_hist[2] = {0};
                int32_t hist_bin = (packet_status == 0) ? 0 : 1;
                s_tx_done_hist[hist_bin] = s_tx_done_hist[hist_bin]+1;
                debug_printf("\t Success = %d, Fail = %d\n", s_tx_done_hist[0], s_tx_done_hist[1]);
                #endif

                // Determine if this is the Tx Status for the mailbox check
                bool mailbox_check = (packet_id == s_pkt_id_of_mailbox_request);
                
                // Send Tx Done status up to caller
                if ((!mailbox_check) && (p_task_cfg->lte_cb_on_tx_done != NULL))
                {
                    p_task_cfg->lte_cb_on_tx_done(packet_status);
                }
                else if ((mailbox_check) && (p_task_cfg->lte_cb_on_mailbox_check_done != NULL))
                {
                    p_task_cfg->lte_cb_on_mailbox_check_done(packet_status);
                }
            }
            else
            {
//               debug_printf("No new uplink packet status\n");
            }
        }
    } while (new_status == 1);
    return 0;
}

int32_t lte_module_send_mailbox_check(void)
{
    uint32_t packet_id = 0xFFFFFFFF;
    int32_t i32_ret = lte_ifc_mailbox_request(&packet_id);
    print_lte_ifc_error("lte_ifc_mailbox_request", i32_ret);
    if (i32_ret == LTE_IFC_ACK)
    {
        // debug_printf("lte_mailbox_request: packet_id = %d\n", packet_id);
        s_pkt_id_of_mailbox_request = packet_id;
    }
    return i32_ret;
}

int32_t lte_module_get_downlink_message(lte_task_config_t* p_task_cfg)
{
    // Get downlink message module, there might be more than 1 (someday)
    
    static uint8_t buff[128];
    uint16_t bytes_received;
    uint8_t port;
    int32_t i32_ret;
    do
    {
        bytes_received = sizeof(buff);
        port = 0;
        
        i32_ret = lte_ifc_retrieve_message(buff, &bytes_received, &port);
        print_lte_ifc_error("lte_ifc_retrieve_message", i32_ret);
        if ((bytes_received > 0) && (p_task_cfg->lte_cb_on_rx_done))
        {
            p_task_cfg->lte_cb_on_rx_done(buff, bytes_received);
        }
    } while (bytes_received > 0);
    return 0;
}
