
#include <atmel_start.h>
#include <string.h>

#include "FreeRTOS.h"
#include "task.h"

#include "lte_ifc.h"
#include "lte_ifc_task.h"
#include "lte_module.h"

#include "debug_print.h"

static TaskHandle_t s_lte_ifc_task_handle = NULL;
static xQueueHandle s_user_tx_queue = NULL;
static xQueueHandle s_cmd_queue;

typedef enum
{
    LTE_CMD_CHECK_MAILBOX
} lte_cmd_t;

typedef struct
{
    uint16_t     len;                               // length of message to be sent
    uint8_t      data[MAX_USER_UPLINK_LENGTH_BYTES];     // message buffer
} tx_msg_struct_t;

static lte_task_config_t s_task_cfg = {0};
void lte_task_update_config(lte_task_config_t* p)
{
    memcpy(&s_task_cfg, p, sizeof(lte_task_config_t));
}

lte_task_ret_t lte_send_msg(const uint8_t* payload, uint16_t len)
{
    if (payload == NULL)
    {
        return LTE_TASK_SEND_MSG_INVALID_BUF;
    }
    if (len >= MAX_USER_UPLINK_LENGTH_BYTES)
    {
        return LTE_TASK_SEND_MSG_INVALID_LEN;
    }
    
    tx_msg_struct_t msg;
    msg.len = len;
    memcpy(msg.data, payload, len);
    
    if (xQueueSendToBack(s_user_tx_queue, &msg, 0) == pdTRUE)
    {
        return (LTE_TASK_SUCCESS);
    }
    else
    {
        return (LTE_TASK_QUEUE_FAIL);
    }
}

lte_task_ret_t lte_downlink_check(void)
{
    lte_cmd_t cmd = LTE_CMD_CHECK_MAILBOX;
    if (xQueueSendToBack(s_cmd_queue, &cmd, 0) == pdTRUE)
    {
        return (LTE_TASK_SUCCESS);
    }
    else
    {
        return (LTE_TASK_QUEUE_FAIL);
    }        
}

static void process_tx_queue(void)
{
    static tx_msg_struct_t msg;
    
    if (xQueuePeek(s_user_tx_queue, &msg, 0) == pdTRUE)
    {
        // There is a message in the queue. It was copied to 'msg'

        int32_t i32_ret = 0;
        i32_ret = lte_module_send_uplink(msg.data, msg.len);
        if (i32_ret == 0)
        {
            // Pop message off of queue
            BaseType_t pop_success = xQueueReceive(s_user_tx_queue, &msg, 0);
            ASSERT(pop_success == pdTRUE);
        }
        // else leave message on queue
    }
}

static void process_cmd_queue(void)
{
    static lte_cmd_t cmd;
    
    if (xQueuePeek(s_cmd_queue, &cmd, 0) == pdTRUE)
    {
        // There is command in the queue. It was copied to 'cmd'

        // only 1 command so far
        ASSERT(cmd == LTE_CMD_CHECK_MAILBOX);
        
        int32_t i32_ret = lte_module_send_mailbox_check();
        if (i32_ret == 0)
        {
            // Pop message off of queue
            BaseType_t pop_success = xQueueReceive(s_cmd_queue, &cmd, 0);
            ASSERT(pop_success == pdTRUE);
            // debug_printf("Mailbox check command pulled off of cmd queue\n");
        }
        // else leave message on queue
    }
    
}

static void process_tx_done(void)
{
    int32_t i32_ret = lte_module_get_uplink_status(&s_task_cfg);
    (void)i32_ret;
}

static void process_rx_done(void)
{
    int32_t i32_ret = lte_module_get_downlink_message(&s_task_cfg);
    (void)i32_ret;
}

portTASK_FUNCTION(lte_ifc_task, param)
{
    (void)param;
    int32_t i32_ret = 0;
    
    i32_ret = lte_module_init();
    ASSERT(i32_ret == 0);

    i32_ret = lte_module_reset();
    ASSERT(i32_ret == 0);

    
    int32_t task_loop_count = 0;
    uint32_t irq_flags = 0;
    while(1)
    {
        if (lte_module_get_irq_flags(&irq_flags) != 0)
        {
            continue;
        }
        
        process_tx_queue();
        process_cmd_queue();
        
        if (irq_flags & IRQ_FLAGS_TX_DONE)
        {
            process_tx_done();
        }
        
        if (irq_flags & IRQ_FLAGS_RX_DONE)
        {
            process_rx_done();
        }
                
        task_loop_count++;
        vTaskDelay(1000 / portTICK_PERIOD_MS);
    }
}

void lte_ifc_task_init(void)
{
    s_user_tx_queue = xQueueCreate(1, sizeof(tx_msg_struct_t));
    ASSERT(s_user_tx_queue != NULL);

    s_cmd_queue = xQueueCreate(5, sizeof(lte_cmd_t));
    ASSERT(s_cmd_queue != NULL);
    
    xTaskCreate(lte_ifc_task, (const portCHAR *)"LTE_IFC", 512, NULL, 1, &s_lte_ifc_task_handle);
}


