#ifndef __LTE_IFC_H
#define __LTE_IFC_H

#include "lte_ifc_bytedefs.h"
#include "lte_ifc_irq_flags.h"
#include "lte_ifc_opcodes.h"
#include "lte_ifc_hal.h"

#ifdef __cplusplus
extern "C" {
#endif

#define LTE_TTY_DEFAULT_BAUDRATE 115200

/**
* @addtogroup Link_Labs_LTE_Interface_Library lte_ifc
*
* @brief External host library which simplifies communication with an LTE Cat-M1
*      network using a Link Labs module.
* 
* This library runs on an external host processor which can be nearly any
* microcontroller or PC with a UART interface.  The library
* is written in C and contains two categories of functions:
* -# @ref lte_ifc_common These are portable functions,
* which are meant to run on any platform. These should be compiled as is with no modifications. These
* functions are the API which you should call from your application.
*
* -# @ref lte_ifc_hal A set of function prototypes for platform-specific and low-level operations.
* These are prototypes only - you must implement your own version of these
* and link them in. Usually these are extremely simple operations, like setting an I/O line high or low,
* or queueing some bytes for transmission through a UART, etc.
* 
* A block diagram of how the two categories fit into an integrated system is shown below.
* \htmlonly <style>div.image img[src="lte_ifc_block.png"]{width:400px;}</style> \endhtmlonly 
* @image html lte_ifc_block.png 
* 
* @{
*/

/** @addtogroup lte_ifc_common lte_ifc_common
 *  @{
 */

/**
 * @brief
 *   Force the module to reset and enter bootloader mode (takes a few seconds).
 *
 * @details
 *   This function forces the module to reset and enter bootloader mode. The module only resets
 *   and enters bootloader mode if the success code is returned.
 *
 * @return
 *   @ref lte_ifc_ack_code
 */
int32_t lte_ifc_bootloader_mode(void);

/**
* @brief
*   Gets assert info from the module, if any is available.
*
* @return
*   @ref lte_ifc_ack_code
*/
int32_t lte_ifc_get_assert_info(char *filename, uint32_t* p_line, uint32_t* p_uptime_sec);

/**
 *@brief
 *  Reads out the IMEI of the device.
 *
 * @param[out] p_imei
 * The IMEI is returned in hex format. The IMEI is 15 decimal digits, so when represented in hex
 *   the length will be 13 hex digits. For example, an IMEI of 357353080026273 will be returned
 *   in uint64_t format as 0x00014502bf64f4a1.
 *
 * @return
 *   @ref lte_ifc_ack_code
 */
int32_t lte_imei_get(uint64_t* p_imei);


/**
 *@brief
 *  Commands the module to check the downlink mailbox. This triggers an uplink transmission containing
 *  the mailbox request.
 *
 * @param[out] p_packet_id
 * If the success code is returned, this address will be populated with a packet id for the mailbox request.
 *
 * @return
 *   @ref lte_ifc_ack_code
 */
int32_t lte_ifc_mailbox_request(uint32_t* p_packet_id);

/**
 * @brief
 *   Request the module to send an uplink message.
 *
 * @param[in] buf
 *   A byte array containing the data payload.
 *
 * @param[in] len
 *   Length of the input buffer in bytes. Must be less than or equal to
 *   MAX_USER_UPLINK_LENGTH_BYTES.
 *
 * @param[in] port
 *   Reserved for future use.
 *
 * @param[out] p_packet_id
 *   The address pointed to by this pointer will be populated with a packet id for the uplink message.
 *
 * @return
 *   @ref lte_ifc_ack_code Note that the success code only indicates if the uplink message was succesfully
 *   queued for transmission, not whether it was successfully transmitted. In order to check the delivery status
 *   the @ref lte_ifc_message_status function should be called.
 */
int32_t lte_ifc_message_send(uint8_t* buf, uint16_t len, uint8_t port, uint32_t* p_packet_id);

/**
 * @brief
 *   Request Check for a status update on pending uplink transmissions.
 *
 * @param[out] p_new_status
 *   This will be set to 1 if a new packet status is available, 0 otherwise.
 *
 * @param[out] p_packet_id
 *   The packet id that the status applies to. Only set if new_status is 1.
 *
 * @param[out] p_packet_status
 *  @ref lte_ifc_ack_code indicating the result of the transmission attempt on the packet. Only set if new_status is 1.
 *
 * @param[out] p_packet_time_in_queue_ms
 *   How long (in milliseconds) the packet was queued in the module, before it was able to be sent. Only set if new_status is 1.
 *
 * @return
 *   @ref lte_ifc_ack_code
 */
int32_t lte_ifc_message_status(uint8_t* p_new_status, uint32_t* p_packet_id, int32_t* p_packet_status, uint32_t* p_packet_time_in_queue_ms);

/**
 * @brief
 *   Get the module firmware version number.
 *
 * @param[inout] version
 *   Pass in a pointer to a lte_version_t data type. This data structure will be populated if the ack code indicates success.
 *
 * @return
 *   @ref lte_ifc_ack_code
 */
int32_t lte_ifc_version_get(lte_version_t *version);

 /**
 * @brief
 *   Save configurable settings to flash.
 *
 * @return
 *   @ref lte_ifc_ack_code
 */
int32_t lte_ifc_settings_save(void);

/**
 * @brief
 *   Delete all settings from flash storage, restoring factory defaults on all configurable values.
 *
 * @return
 *   @ref lte_ifc_ack_code
 *
 */
int32_t lte_ifc_settings_delete(void);

/**
 * @brief
 *   Force the module to reset (takes a few seconds).
 *
 * @details
 *   This function forces the module to reset, if possible. The module only resets if a success code is returned.
 *   If any other code is returned, the reset request was rejected. In this case, the function should be called
 *   again some time (e.g. 1 second) later until  a success code is returned.
 *
 * @return
 *   @ref lte_ifc_ack_code
 */
int32_t lte_ifc_reset_mcu(void);


/**
* @brief
*   Triggers an assert (for test purposes only).
*
* @return
*   @ref lte_ifc_ack_code
*   */
int32_t lte_ifc_trigger_assert(void);

/**
* @brief
*   Set received mode.
*
* @param[in] rx_mode
*   Reserved for future use.
*
* @return
*   @ref lte_ifc_ack_code
*   */
int32_t lte_ifc_receive_mode_set(uint8_t rx_mode);

/**
* @brief
*   Get received mode.
*
* @param[out] p_rx_mode
*   Reserved for future use.
*
* @return
*   @ref lte_ifc_ack_code
*   */
int32_t lte_ifc_receive_mode_get(uint8_t *p_rx_mode);

/**
 * @brief
 *   Retrieves a downlink message received by the module.
 *
 * @param[out] buf
 *   The buffer into which the received message will be placed. This buffer must
 *   be at least 1 byte larger than the maximum message size expected.
 *
 * @param[in,out] size
 *   The size of the output buffer. If the message is successfully retrieved,
 *   this will be set to the size of the message (not including the port).
 *
 * @param[out] port
 *   The port number associated with the message.
 *
* @return
*   @ref lte_ifc_ack_code
 */
int32_t lte_ifc_retrieve_message(uint8_t *buf, uint16_t *size, uint8_t *port);

/**
 * @brief
 *   Retrieves a firmware-over-the-air image.
 *
 * @param[in] filename
 *   The ASCII filename of the file to download. Does not need to be NULL-terminated,
 *   since the length is also passed in. Can only contain either alphanumeric
 *   characters or a character in the set {-._}.
 *
 * @param[in] len
 *   Length of the filename (number of ASCII chars). This is needed since the 
 *   filename argument is not NULL-terminated.
 *
 * @return
 *   @ref lte_ifc_ack_code
 */
int32_t lte_ifc_fota_download(uint8_t *filename, uint16_t len);

/**
 * @brief
 *   Retrieves the state of a firmware-over-the-air download.
 *
 * @param[out] p_state
 *   The state of the FOTA process.
 *
 * @param[out] p_end_reason
 *   The result of the last run FOTA process since power-on reset.
 *
 * @param[out] p_num_blocks_received
 *   The number of blocks received.
 *
 * @param[out] p_num_requests_sent
 *   The number of file requests sent.
 *
 * @param[out] p_transfer_complete
 *   Whether the transfer is complete or not.
 *
 * @return
 *   @ref lte_ifc_ack_code
 */

int32_t lte_ifc_fota_get_state(uint8_t* p_state,
                               uint8_t* p_end_reason,
                               uint32_t* p_num_blocks_received,
                               uint32_t* p_num_requests_sent,
                               bool* p_transfer_complete);


/**
 * @brief
 *   Adds an ASCII message to the debug log.
 *
 * @param[in] chars
 *   ASCII chars that make up the 'message'. Only printable characters
 *   are accepted: must be in range (0x20 <= x <= 0x7E).
 *
 * @param[in] len
 *   Length of the message (number of ASCII chars). This is needed since the
 *   message argument is not NULL-terminated. Max length is 32 chars.
 *
 * @return
 *   @ref lte_ifc_ack_code
 */
int32_t lte_ifc_debug_msg(uint8_t *chars, uint16_t len);

/**
* @brief
*   Set how often the module checks for the availability of a FOTA
*   image.
*
* @param[in] period
*   Check period, in minutes. All uint16_t values are allowed:
*   0 = FOTA check is disabled
*   1-65535 = FOTA check every 1 to 65,535 minutes.
*
* @return
*   @ref lte_ifc_ack_code
*   */
int32_t lte_ifc_fota_check_period_set(uint16_t period);

/**
* @brief
*   Get fota check period.
*
* @param[out] p_period
*   See set command for parameter details.
*
* @return
*   @ref lte_ifc_ack_code
*   */
int32_t lte_ifc_fota_check_period_get(uint16_t *p_period);

/**
* @brief
*   Set when the module checks for the availability of a FOTA
*   image.
*
* @param[in] check_at_boot
*   Should a check be performed when the Link Labs MCU is reset?
*   0 = No
*   1 = Yes
*
* @return
*   @ref lte_ifc_ack_code
*   */
int32_t lte_ifc_fota_check_at_boot_set(uint8_t check_at_boot);

/**
* @brief
*   Get when the module checks for the availability of a FOTA image.
*
* @param[out] p_check_at_boot
*   See set command for parameter details.
*
* @return
*   @ref lte_ifc_ack_code
*   */
int32_t lte_ifc_fota_check_at_boot_get(uint8_t *p_check_at_boot);

/**
* @brief
*   Set the module's power policy.
*
* @param[in] pwr_policy
*   0 = On/Off with inactivity timer
*   1 = LTE Cat-M1 Power Savings Mode (PSM)
*
* @return
*   @ref lte_ifc_ack_code
*   */
int32_t lte_ifc_pwr_policy_set(uint8_t pwr_policy);

/**
* @brief
*   Get the module's power policy.
*
* @param[out] p_pwr_policy
*   See set command for parameter details.
*
* @return
*   @ref lte_ifc_ack_code
*   */
int32_t lte_ifc_pwr_policy_get(uint8_t *p_pwr_policy);

/** @addtogroup lte_ifc_ack_code lte_ifc_ack_code
 *  @{
 */

/**
 * @brief
 *   A short string providing human readable text describing an ack code.
 *
 * @param[in] ack_code
 *   The ack code.
 *
 * @return
 *   The string description.
 */
const char* lte_ifc_ack_desc(int32_t ack_code);

/** @} (end addtogroup lte_ifc_ack_code) */

/** @addtogroup lte_ifc_irq_flags lte_ifc_irq_flags
 *  @{
 */

/**
 * @brief
 *   A short string providing human readable text describing an irq flag.
 *
 * @param[in] irq_flag_bit
 *    Since only one string can be returned at a time, pass in only one
 *    irq flag bit at a time.
 *
 * @return
 *   The string description.
 */
const char* lte_ifc_irq_flag_desc(uint32_t irq_flag_bit);

/**
 * @brief
 *   Read (and optionally clear) IRQ flags in module.
 *
 * @details
 *   This function allows the external host to check whether an event has occurred in the
 *   module that has latched a bit in the "IRQ Flags" vector.
 *
 * @param[in] flags_to_clear
 *   A uint32_t bit vector containing flags that should be cleared if they are set. This can be
 *   0 if the host interface just wants to read without clearing. If a bit is set, this function
 *   performs a clear-on-read of the irq_flags bits passed in.
 *
 * @param[out] flags
 *   A uint32_t bit vector - the value of the irq_flags in the module. Note that if the flags_to_clear
 *   argument is non-zero, this argument is the value of the flags before the clear operation.
 *
 * @return
 *   0 - success, negative otherwise.
 */
int32_t lte_ifc_irq_flags(uint32_t flags_to_clear, uint32_t *flags);

/** @} (end addtogroup lte_ifc_irq_flags) */

/** @} (end addtogroup lte_ifc_common) */

/** @} (end addtogroup Link_Labs_LTE_Interface_Library) */

#ifdef __cplusplus
}
#endif

#endif /* __LTE_IFC_H */
