#include <atmel_start.h>
#include <stdbool.h>
#include <stdint.h>

#include "FreeRTOS.h"
#include "task.h"

#include "lte_ifc_task.h"
#include "debug_print.h"

#define SW_VERSION_MAJOR    (1)
#define SW_VERSION_MINOR    (0)

#define DO_TX
// #define DO_RX

// Tx vars
static uint32_t s_pkt_count = 0;
// static uint32_t s_round_robin_pkt_intervals_s[] = {0, 0};    // Tx Only "Always On Endurance Test"
// static uint32_t s_round_robin_pkt_intervals_s[] = {15, 15};  // Tx Only "On-Off Endurance Test"
static uint32_t s_round_robin_pkt_intervals_s[] = {0, 15};  // Tx "Mixed Test"    
volatile TickType_t s_next_tx_tick = 0;
static const TickType_t S_TX_TIMEOUT_TICKS = pdMS_TO_TICKS(60*1000);
volatile TickType_t s_tx_timeout_tick = 0;
static bool s_pkt_sent = false;

// Rx vars
static const TickType_t s_rx_interval_ticks = pdMS_TO_TICKS(60*1000);
static TickType_t s_next_rx_tick = 0;

static void my_tx_done(int32_t status)
{
    debug_printf("app: my_tx_done %d\n", status);

    s_pkt_sent = false;
    s_pkt_count++;
    s_next_tx_tick = xTaskGetTickCount();
    uint32_t idx = (s_pkt_count&1) ? 0 : 1;
    s_next_tx_tick += pdMS_TO_TICKS(s_round_robin_pkt_intervals_s[idx]*1000);
}

static void my_mailbox_check_done(int32_t status)
{
    debug_printf("app: my_mailbox_check_done %d\n", status);
}

static void my_rx_done(uint8_t* buf, uint16_t len)
{
    ASSERT(buf != NULL);
    debug_printf("app: my_rx_done 0x%08X %d\n", buf, len);
    for (uint16_t n=0; n<len; n++)    
    {
        debug_printf("0x%02X ", buf[n]);
    }
    debug_printf("\n");
}

static TaskHandle_t s_demo_app_task_handle = NULL;
portTASK_FUNCTION(demo_app_task, param)
{
    (void)param;

    lte_task_config_t my_cfg;
    my_cfg.lte_cb_on_tx_done = my_tx_done;
    my_cfg.lte_cb_on_mailbox_check_done = my_mailbox_check_done;
    my_cfg.lte_cb_on_rx_done = my_rx_done;
    lte_task_update_config(&my_cfg);

    debug_printf("Link Labs LTE-M Xplained Demo v%d.%d\n", SW_VERSION_MAJOR, SW_VERSION_MINOR);

    while(1)
    {
        vTaskDelay(1000 / portTICK_PERIOD_MS);
        
        #ifdef DO_RX
        // Check if time to Rx
        if (((int32_t)(xTaskGetTickCount() - s_next_rx_tick)) >= 0)
        {
            debug_printf("app: Sending dl check from demo app\n");
            lte_downlink_check();
            s_next_rx_tick += s_rx_interval_ticks;
        }
        #else
        (void)s_next_rx_tick; // Suppress warning
        #endif
        
        #ifdef DO_TX
        // Check if time to Tx: 
        // 1) Either we're idle and its time to tx
        // 2) We're waiting for tx_done and timeout expires
        if (((!s_pkt_sent) && (((int32_t)(xTaskGetTickCount() - s_next_tx_tick)) >= 0))  ||
            ((s_pkt_sent) && (((int32_t)(xTaskGetTickCount() - s_tx_timeout_tick)) >= 0)))
        {
            static uint32_t s_ul_payload = 0;
            s_ul_payload++;
            uint16_t len = 4;

            debug_printf("s_ul_payload = 0x%08X\n", s_ul_payload);
            if (lte_send_msg((const uint8_t*)&s_ul_payload, len) == LTE_TASK_SUCCESS)
            {
                s_pkt_sent = true;
            }
            else
            {
                debug_printf("ERROR: unable to queue send msg\n");
            }
            // Set timeout so we dont wait forever in case we never get tx_done() call
            s_tx_timeout_tick = xTaskGetTickCount() + S_TX_TIMEOUT_TICKS;
        }
        #endif        
    }
}


void demo_app_task_init(void)
{
    xTaskCreate(demo_app_task, (const portCHAR *)"DEMO_APP", 256, NULL, 1, &s_demo_app_task_handle);
}


