#! /usr/bin/env python

"""
This script loads encrypted firmware onto multiple modules through the bootloader interface.
"""

import logging
from xmodem import XMODEM, CRC
from time import sleep, time
import argparse
from multiprocessing.pool import ThreadPool
from functools import partial

import ll_ifc

LOG = logging.getLogger(__name__)

def upload_firmware(dev, firmware_path):
    LOG.info("Uploading firmware to device %s", dev.sdev.port)

    LOG.debug("Putting module into bootloader mode")
    dev.reboot_into_bootloader()

    LOG.debug("Looking for the bootloader menu")
    dev.sdev.write(b'h')
    sleep(1.0)
    while dev.sdev.inWaiting():
        if b'r - reset' in dev.sdev.readline():
            break
    else:
        raise RuntimeError("Couldn't find bootloader menu")

    def getc(size, timeout=1):
        return dev.sdev.read(size)

    def putc(data, timeout=1):
        dev.sdev.write(data)

    LOG.debug("Sending firmware via XMODEM")
    with open(firmware_path, 'rb') as firmware:
        dev.sdev.write(b'u')

        LOG.debug("Waiting for CRC character")
        start = time()
        while time() - start < 3.0:
            char = dev.sdev.read(1)
            if char == CRC:
                break
        else:
            raise RuntimeError("Timeout waiting for CRC character")

        LOG.debug("Starting XMODEM transfer.")
        upload_success = XMODEM(getc, putc).send(firmware, retry=16, timeout=60*4, quiet=0)

        if upload_success:
            LOG.debug("Firmware uploaded to %s", dev.sdev.port)
            verify_firmware(dev)
        else:
            raise RuntimeError("Error in XMODEM transfer")

        # Reset module into firmware
        LOG.info("New firmware uploaded and verified by module %s", dev.sdev.port)
        dev.sdev.write(b'r')


def verify_firmware(dev):
    """
    Commands the bootloader to verify firmware, then asserts that the firmware is valid.
    Note: the module must be in bootloader mode when this function is called.
    """
    dev.sdev.write(b'v')
    sleep(10)
    while dev.sdev.inWaiting():
        line = dev.sdev.readline()
        if b'Verifying...OK' in line:
            break
        elif b'Verifying...INVALID' in line:
            raise RuntimeError("New firmware not valid.")
    else:
        raise RuntimeError("Never got new firmware verification status.")


def main():
    parser = argparse.ArgumentParser(description=__doc__)
    parser.add_argument('firmware', help='path to the encrypted firmware to load')
    parser.add_argument('--device', help='Path of the device to program. \
        Default is to find CP210x devices and upload to all of them.')
    parser.add_argument('--verbose', '-v', action='store_true')
    args = parser.parse_args()

    if args.verbose:
        LOG.setLevel(logging.DEBUG)

    if args.device:
        with ll_ifc.ModuleConnection(args.device) as dev:
            upload_firmware(dev, args.firmware)
    else:
        with ll_ifc.get_all_modules() as devs:
            upload_with_firmware = partial(upload_firmware, firmware_path=args.firmware)
            pool = ThreadPool(len(devs))
            pool.map(upload_with_firmware, devs)


if __name__ == '__main__':
    logging.basicConfig(level=logging.INFO)
    main()
